<?php

require_once(__DIR__ . "/vendor/autoload.php");

class API
{
    protected string $baseUri    = "https://toybox.maxweb.cloud";
    protected string $apiVersion = "v2";
    protected string $error;

    public function __construct($baseUri = null)
    {
        if ($baseUri !== null) {
            $this->setBaseUri($baseUri);
        }
    }

    /**
     * Detect if we are connected to the Toybox API - it's not a live check,
     * but as tokens shouldn't really expire, we can set an option inside of
     * WordPress itself and check for its existence, and if it exists we know
     * we've logged in at least once before. We'll do an additional check at a
     * later point (before we do anything major over the API) to ensure we
     * don't encounter any issues.
     *
     * @return bool
     */
    public function isConnected(): bool
    {

        $token = $this->getToken();

        if ($token === false || ! $this->verifyToken($token)) {
            return false;
        }

        return true;
    }

    /**
     * Get the API base URI.
     *
     * @return string
     */
    public function getBaseUri(): string
    {
        return $this->baseUri;
    }

    /**
     * Set the API base URI.
     *
     * @param string $baseUri
     *
     * @return void
     */
    public function setBaseUri(string $baseUri): void
    {
        $this->baseUri = $baseUri;
    }

    /**
     * Get the API version.
     *
     * @return string
     */
    public function getApiVersion(): string
    {
        return $this->apiVersion;
    }

    /**
     * Set the API version.
     *
     * @param string $version
     *
     * @return void
     */
    public function setApiVersion(string $version): void
    {
        $this->apiVersion = $version;
    }

    public function verifyToken(string $token): bool
    {
        $response = $this->request("POST", $this->getEndpoint("/verifyToken"), ["token" => $token]);

        return $response->success === true;
    }

    /**
     * Attempt login to the API.
     *
     * @param string $email
     * @param string $password
     *
     * @return bool
     * @throws Exception
     */
    public function authenticate(string $email, string $password): bool
    {
        $login = $this->request("POST", $this->getEndpoint("/login"), ["email" => $email, "password" => $password]);

        if (property_exists($login, "error")) {
            $this->error = $login->error;

            return false;
        }

        if (property_exists($login, "token")) {
            $this->setToken($login->token);
        }

        // Create a site (if we have one, just store its site ID locally)
        $site = $this->createSite();
        $this->setSite($site);

        return true;
    }

    public function createSite()
    {
        $name = get_option("blogname");

        $site = $this->authRequest("POST", $this->getEndpoint("/sites/create"), ["name" => $name]);

        return $site;
    }

    public function createEnvironment(string $name): bool
    {
        global $wp_version;

        $site    = $this->getSite();
        $home    = get_option("home");
        $siteURL = get_option("siteurl");
        $plugins = get_plugins();

        $response = $this->authRequest("POST", $this->getEndpoint("/sites/{$site->id}/environments/create"), [
            "name"              => $name,
            "home"              => $home,
            "siteurl"           => $siteURL,
            "wordpress_version" => $wp_version,
            "plugins"           => json_encode($plugins),
        ]);

        if (property_exists($response, "success")) {
            if ($response->success === true) {
                $this->setEnvironment($response->environment);
            }

            return $response->success;
        }

        return false;
    }

    private function getEndpoint(string $uri): string
    {
        if (! str_starts_with($uri, "/")) {
            $uri = "/{$uri}";
        }

        return "{$this->getBaseUri()}/api/{$this->getApiVersion()}{$uri}";
    }

    /**
     * @throws Exception
     */
    public function authRequest(string $method, string $endpoint, array $data = [], array $headers = [])
    {
        $token = $this->getToken();

        return $this->request($method, $endpoint, $data, array_merge(["Authorization" => "Bearer {$token}"], $headers));
    }

    /**
     * @throws Exception
     */
    public function request(string $method, string $endpoint, array $data = [], array $headers = [])
    {
        // dump($endpoint, $data, $headers);

        $method = strtoupper($method);

        // Setup the Guzzle client
        $client = new \GuzzleHttp\Client();

        // Setup options
        $options = [];

        // Set headers
        if (! empty($headers)) {
            $options["headers"] = $headers;
        }

        // Setup the request
        if ($method === "GET") {
            if (! empty($data)) {
                $options["query"] = $data;
            }
        } else {
            if (! empty($data)) {
                foreach ($data as $key => $value) {
                    $options["multipart"][] = [
                        "name"     => $key,
                        "contents" => $value,
                    ];
                }
            }
        }

        $response = $client->request($method, $endpoint, $options);

        $json = json_decode($response->getBody());

        if ($json !== null) {
            return $json;
        }

        return $response->getBody();
    }

    public function sync(): void
    {
        $token = $this->getToken();

        // Firstly, verify the token
        if (! $this->verifyToken($token)) {
            $this->deleteToken();
            $this->deleteSite();
            $this->deleteEnvironment();

            return;
        }

        // Token was verified, re-sync the site
        $site = $this->createSite();
        $this->setSite($site);

        // Re-sync the environment (if one is configured)
        if ($environment = $this->getEnvironment()) {
            $this->createEnvironment($environment->name);
            $this->checkIn();
        }
    }

    public function checkIn(): bool
    {
        global $wp_version;

        if (! function_exists("get_plugins")) {
            require_once(ABSPATH . "wp-admin/includes/plugin.php");
        }

        if (! function_exists("get_plugin_updates")) {
            require_once(ABSPATH . "wp-admin/includes/update.php");
        }

        $environment   = $this->getEnvironment();
        $plugins       = get_plugins();
        $pluginUpdates = [];

        foreach (get_plugin_updates() as $plugin) {
            $pluginUpdates[$plugin->Name] = [
                "old" => $plugin->Version,
                "new" => $plugin->update->new_version,
            ];
        }

        $response = $this->authRequest("POST", $this->getEndpoint("/sites/{$environment->site_id}/environments/{$environment->id}/check-in"), [
            "wordpress_version" => $wp_version,
            "plugins"           => json_encode($plugins),
            "plugin_updates"    => json_encode($pluginUpdates),
        ]);

        if (property_exists($response, "success")) {
            return $response->success;
        }

        return false;
    }

    public function getToken(): string
    {
        return get_option("toybox_api_token");
    }

    public function setToken(string $token): bool
    {
        return update_option("toybox_api_token", $token, true);
    }

    public function deleteToken(): bool
    {
        return delete_option("toybox_api_token");
    }

    public function getSite(): object
    {
        return get_option("toybox_site");
    }

    public function setSite(object $site): void
    {
        update_option("toybox_site", $site, false);
    }

    public function deleteSite(): bool
    {
        return delete_option("toybox_site");
    }

    public function getEnvironment(): object
    {
        return get_option("toybox_site_environment");
    }

    public function setEnvironment(object $environment): void
    {
        update_option("toybox_site_environment", $environment, false);
    }

    public function deleteEnvironment(): bool
    {
        return delete_option("toybox_site_environment");
    }

    public function getBlocks(int $page = 1)
    {
        $q = ! empty($_GET["q"]) ? $_GET["q"] : "";

        $response = $this->authRequest("GET", $this->getEndpoint("/blocks"), [
            "page" => $page,
            "q"    => $q,
        ]);

        return $response;
    }

    public function exportBlock(string $title, string $description, string $version, string $fileName): bool
    {
        $response = $this->authRequest("POST", $this->getEndpoint("/blocks/export"), [
            "name"        => $title,
            "description" => $description,
            "zip"         => \GuzzleHttp\Psr7\Utils::tryFopen($fileName, "r"),
            "version"     => $version,
        ]);

        // dump($response);

        if ($response && is_object($response) && property_exists($response, "success")) {
            return (bool) $response->success;
        }

        return false;
    }

    public function downloadBlock(int $blockID, string $slug, string $version = "latest")
    {
        $response = $this->authRequest("GET", $this->getEndpoint("/blocks/{$blockID}"), [
            "version" => $version,
        ]);

        if ($response) {
            $zip = $response;

            if (! file_exists(get_theme_file_path("blocks"))) {
                mkdir(get_theme_file_path("blocks"));
            }

            if (! file_exists(get_theme_file_path("blocks/{$slug}"))) {
                mkdir(get_theme_file_path("blocks/{$slug}"));
            }

            $fh = fopen(get_theme_file_path("blocks/{$slug}/block.zip"), "w");
            fwrite($fh, $zip);
            fclose($fh);

            // Unzip it
            $path = get_theme_file_path("blocks/{$slug}");

            $zip = new \PhpZip\ZipFile();

            try {
                $zip->openFile(get_theme_file_path("blocks/{$slug}/block.zip"))
                    ->extractTo($path);
            } catch (\PhpZip\Exception\ZipException $e) {
                // Fail silently
            } finally {
                $zip->close();
            }

            // Delete the zip
            unlink(get_theme_file_path("blocks/{$slug}/block.zip"));

            // Import ACF fields
            if (function_exists("acf_import_field_group")) {
                acf_import_field_group(json_decode(file_get_contents(get_theme_file_path("blocks/{$slug}/acf-json/block-{$slug}.json")), true));
            }

            return true;
        }

        return false;
    }

    public function versionExists($name, $version)
    {
        $response = $this->authRequest("POST", $this->getEndpoint("/blocks/versionCheck"), [
            "name"    => $name,
            "version" => $version,
        ]);

        if ($response && is_object($response) && property_exists($response, "success")) {
            return (bool) $response->success === false;
        }

        return false;
    }
}
