<?php

class Maintainer
{
    public API $api;

    public function __construct()
    {
        require_once(__DIR__ . "/API.php");

        $this->api = new API();
    }

    public function showApiPage(): void
    {
        if (! $this->api->isConnected()) {
            $this->showApiLoginPage();

            return;
        }

        $this->showApiDashboard();
    }

    public function showApiDashboard(): void
    {
        if (! empty($_GET["action"]) && $_GET["action"] === "sync") {
            $this->api->sync();
        }

        echo <<<EOD
            <h2>Toybox API <small>CONNECTED</small></h2>
            
            <a class="button" href="?page=toybox&area=api&action=sync">Sync with API</a>

            <style>
                .toybox-wrapper h2 {
                    display: flex;
                    align-items: center;
                    gap: 1rem;
                }
                
                .toybox-wrapper h2 small {
                    background: #44bd32;
                    color: #f5f6fa;
                    font-size: .75rem;
                    line-height: .75rem;
                    padding: .25rem .225rem;
                    border-radius: .25rem;
                }
            </style>
        EOD;

    }

    public function showUpdatesPage(): void
    {
        $updates = "";
        $updates2 = "";

        echo "<h2>Plugins</h2>";

        foreach (get_plugin_updates() as $updatedPlugin) {
            $updates .= "{$updatedPlugin->Name} {$updatedPlugin->Version} → {$updatedPlugin->update->new_version}\r\n";
            $updates2 .= "<p>{$updatedPlugin->Name} {$updatedPlugin->Version} → {$updatedPlugin->update->new_version}</p>";
        }

        $updates = trim($updates);

        echo <<<EOD
            <textarea id="updateList" class="regular-text" style="width: 100%; height: 600px;" readonly>{$updates}</textarea>
            <button class="button" type="button" data-copy-target="#updateList2">Copy to Clipboard</button>
            <div id="updateList2">{$updates2}</div>
            
            <style>
                #updateList2 {
                    display: none;
                }
            </style>

            <script>
                document.addEventListener("DOMContentLoaded", () => {
                    for (let btn of document.querySelectorAll("[data-copy-target]")) {
                        btn.addEventListener("click", () => {
                            let target        = btn.getAttribute("data-copy-target");
                            let originalInner = btn.innerText;
                            
                            if (window.isSecureContext && navigator.clipboard) {
                                navigator.clipboard.writeText(document.querySelector(target).innerText);
                            } else {
                                unsecuredCopyToClipboard(document.querySelector(target).innerText);
                            }
                            
                            btn.setAttribute("disabled", true);
                            btn.innerText = "Copied!";
                            
                            setTimeout(() => {
                                btn.innerText = originalInner;
                                btn.removeAttribute("disabled");
                            }, 3000);
                        });
                    }
                    
                    const unsecuredCopyToClipboard = (text) => {
                        const textArea = document.createElement("textarea");
                        textArea.value = text;
                        
                        document.body.appendChild(textArea);
                        
                        textArea.focus();
                        textArea.select();
                        
                        try {
                            document.execCommand('copy');
                        } catch (err) {
                            console.error('Unable to copy to clipboard', err);
                        }
                        
                        document.body.removeChild(textArea)
                    };
                });
            </script>
        EOD;

    }

    public function showBackupsPage(): void
    {
        echo <<<EOD
            <h2>Backups</h2>
        EOD;

    }

    public function showEnvironmentsPage(): void
    {
        $success = null;

        if (! empty($_GET["action"]) && $_GET["action"] === "add") {
            $success = $this->api->createEnvironment($_POST["name"]);

            echo <<<EOD
                <h2>Environments</h2>

                <div class="alert alert-success">
                    Added the environment successfully.
                </div>
            EOD;

            return;
        }

        // Check if we have an environment
        if ($this->hasStoredEnvironment()) {
            $environment = $this->getStoredEnvironment();

            echo <<<EOD
                <h2>Environments</h2>
                
                <p>Environment Name: {$environment->name}</p>
            EOD;

            return;
        }

        add_thickbox();

        echo "<h2>Environments</h2>";

        echo "<a class='button thickbox' href='?TB_inline&width=500&height=140&inlineId=registerEnvironment'>Register Environment</a>";

        $formNonce = wp_nonce_field('my_custom_form_action', 'my_custom_form_nonce');
        $submit    = get_submit_button('Submit');

        echo <<<EOD
            <div id="registerEnvironment" style="display: none;">
                <div>
                    <form action="tools.php?page=toybox&area=environments&action=add" method="post">
                        $formNonce
                        
                        <table class="form-table">
                            <tr valign="top">
                                <th scope="row">
                                    <label for="name">Name</label>
                                </th>
                                
                                <td>
                                    <input type="text" id="name" name="name" value="" class="regular-text" placeholder="">
                                </td>
                            </tr>
                        </table>
                        
                        {$submit}
                    </form>
                </div>
            </div>
        EOD;
    }

    public function showApiLoginPage()
    {
        $formNonce = wp_nonce_field('toybox_api_login', 'toybox_api_login_nonce');
        $submit    = get_submit_button('Login');
        $login     = false;

        if (! empty($_POST['submit'])) {
            // Try login
            $login = $this->api->authenticate($_POST["input_email"], $_POST["input_password"]);
        }

        if ($login === false) {
            $baseUri = $this->api->getBaseUri();

            // WordPress-y stuff
            $wpHome    = defined("WP_HOME")    ? WP_HOME    : get_option("home");
            $wpSiteUrl = defined("WP_SITEURL") ? WP_SITEURL : get_option("siteurl");

            echo <<<EOD
                <h2>Login to connect to Toybox API</h2>
                <p>Base URI: {$baseUri}</p>
                <p>WP Home: {$wpHome}</p>
                <p>WP Site URL: {$wpSiteUrl}</p>
                
                <form action="tools.php?page=toybox&area=api&action=login" method="post">
                    {$formNonce}
                    <table class="form-table">
                        <tr valign="top">
                            <th scope="row"><label for="input_email">Email</label></th>
                            <td><input type="email" id="input_email" name="input_email" value="" class="regular-text"></td>
                        </tr>
                        
                        <tr valign="top">
                            <th scope="row"><label for="input_password">Password</label></th>
                            <td><input type="password" id="input_password" name="input_password" value="" class="regular-text"></td>
                        </tr>
                    </table>
                    
                    {$submit}
                </form>
            EOD;
        }
    }

    public function hasStoredEnvironment()
    {
        $environment = $this->getStoredEnvironment();

        if ($environment) {
            return true;
        }

        return false;
    }

    public function getStoredEnvironment()
    {
        return get_option("toybox_site_environment");
    }
}
