<?php

if (! class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class Backpack_Schedule_List_Table extends WP_List_Table
{
    public function __construct()
    {
        parent::__construct([
            'singular' => 'task', // Singular name of the item
            'plural'   => 'tasks', // Plural name of the items
            'ajax'     => false, // Set to true if using Ajax
        ]);
    }

    // Define columns
    public function get_columns()
    {
        return [
            'cb'          => '<input type="checkbox" />', // Checkbox for bulk actions
            'name'        => 'Name',
            'type'        => 'Type',
            // 'location'    => 'Location',
            'next_due_at' => 'Next Due',
        ];
    }

    // Default column rendering
    protected function column_default($item, $column_name)
    {
        global $backpack;

        switch ($column_name) {

            case 'name':
                $runURL    = sprintf('tools.php?page=mws-backpack&tab=schedule&run=%s', $item['ID']);
                $editURL   = sprintf('tools.php?page=mws-backpack&tab=schedule&edit=%s', $item['ID']);
                $deleteURL = sprintf('tools.php?page=mws-backpack&tab=schedule&delete=%s', $item['ID']);
                $unlockURL = sprintf('tools.php?page=mws-backpack&tab=schedule&unlock=%s', $item['ID']);

                $template = "
                    <strong><a href='%s'>%s%s</a></strong>
                    <div class='row-actions'>
                        %s
                        <span class='run-now'>
                            <a href='%s'>Run Now</a>
                        </span> |
                        
                        <span class='edit'>
                            <a href='%s'>Edit</a>
                        </span> |
                        
                        <span class='trash'>
                            <a href='%s'>Bin</a>
                        </span>
                    </div>
                ";

                return sprintf($template,
                    esc_url($editURL),
                    (((bool) $item["locked"] === true) ? "🔒 " : ""),
                    esc_html(wp_unslash($item[$column_name])),
                    (((bool) $item["locked"] === true) ?
                        sprintf("<span class='unlock'><a href='%s'>Unlock</a></span> |",
                            $unlockURL
                        ) : ""),
                    esc_url($runURL),
                    esc_url($editURL),
                    esc_url($deleteURL),
                );
            case 'type':
                return $backpack->getTaskType($item[$column_name]);

                break;

            case 'location':
                return $item[$column_name];

            case 'next_due_at':
                if ($item[$column_name] !== null) {
                    $date = Carbon\Carbon::createFromFormat("Y-m-d H:i:s", $item[$column_name], $backpack->getTimezone());
                    return $date->format('d/m/Y H:i');
                }

                return "N/A";
            default:
                return print_r($item, true); // Show the array for undefined columns
        }
    }

    // Render the checkbox column
    protected function column_cb($item)
    {
        return sprintf(
            '<input type="checkbox" name="bulk-select[]" value="%s" />',
            $item['ID']
        );
    }

    // Prepare the items to display in the table
    public function prepare_items()
    {
        $columns  = $this->get_columns();
        $hidden   = [];
        $sortable = $this->get_sortable_columns();

        $this->_column_headers = [$columns, $hidden, $sortable];

        // Get sort parameters
        $orderBy = (! empty($_GET['orderby'])) ? sanitize_text_field($_GET['orderby']) : 'name'; // Default sort column.
        $order   = (! empty($_GET['order'])) && 'desc' === strtolower($_GET['order']) ? 'DESC' : 'ASC'; // Default order.


        $data         = $this->get_data($orderBy, $order);
        $current_page = $this->get_pagenum();
        $per_page     = 10;

        $total_items = count($data);
        $data        = array_slice($data, (($current_page - 1) * $per_page), $per_page);

        $this->items = $data;

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page),
        ]);
    }

    private function get_data(string $orderBy = 'name', string $order = 'ASC')
    {
        global $wpdb;

        $table_name = $wpdb->prefix . 'backpack_schedule';

        $query   = "SELECT id as ID, name, type, location, next_due_at, locked FROM $table_name ORDER BY $orderBy $order"; // Match field names to your table schema
        $results = $wpdb->get_results($query, ARRAY_A);

        return $results ? $results : [];
    }

    // Define sortable columns
    protected function get_sortable_columns()
    {
        return [
            'name'        => ['name', true],
            'type'        => ['type', false],
            'location'    => ['location', false],
            'next_due_at' => ['next_due_at', false],
        ];
    }
}
